//=================================================================
// Brdf models
//
// Author : Daniel Meneveaux
// Date   : September 2010
//=================================================================

#include "brdf.hpp"

#include <stdio.h>
#include <iostream>
using namespace std;

static const float INFTY_BRDF=1e20;

//=================================================================
// LAMBERT model
//=================================================================


//=================================================================
float rLambert::fr(pVect rVi, pVect rVo, pVect N)
{
	return m_Kd/M_PI;
}

//=================================================================
void rLambert::resetParam(int i)
{
	m_Kd = 0.0;
}

//=================================================================
void rLambert::changeParam(int i, float val)
{
	m_Kd += val;
	m_Kd=pFloat::min(m_Kd,1.0);
	m_Kd=pFloat::max(m_Kd,EPSILON);
}

//=================================================================
char *rLambert::paramValues()
{
	sprintf(m_values,"Lambert : Kd=%2.2f",m_Kd);
	return m_values;
}


//=================================================================
// MODIFIED PHONG model
//=================================================================


//=================================================================
float rPhong::fr(pVect rVi, pVect rVo, pVect N)
{
	float R;
	pVect vM = rVo-N*((N*rVo)*2.0);
	float scal = (vM*rVi);
	if(scal<EPSILON) scal = 0.0;
	R=m_Kd/M_PI+(m_Ns+2.0)*m_Ks/(2.0*M_PI)*pow(scal,m_Ns);
	return R;
}


//=================================================================
void rPhong::resetParam(int i)
{
	switch(i) {
		case 1:	m_Kd = 0.0; break;
		case 2:	m_Ks = 0.0; break;
		case 3:	m_Ns = 0.0; break;
	}
}

//=================================================================
void rPhong::changeParam(int i, float val)
{
	switch(i) {
		case 1:	m_Kd += val; break;
		case 2:	m_Ks += val; break;
		case 3:	m_Ns += val*200; break;
	}

	m_Kd=pFloat::min(m_Kd,1.0);
	m_Kd=pFloat::max(m_Kd,EPSILON);

	m_Ks=pFloat::min(m_Ks,1.0-m_Kd);
	m_Ks=pFloat::max(m_Ks,EPSILON);

	m_Ns=pFloat::max(m_Ns,0);
}

//=================================================================
char *rPhong::paramValues()
{
	sprintf(m_values,"Phong : Kd=%2.2f   Ks=%2.2f   n=%d",
					m_Kd,m_Ks,m_Ns);
	return m_values;
}


//=================================================================
// COOK-TORRANCE model
//=================================================================

//=================================================================
float rCook::fr(pVect rVi, pVect rVo, pVect N)
{
	pVect L=rVi;
	pVect V=-rVo;
	pVect H=(L+V);
	H.normalize();

	// Estimate D : distribution parameter
	float ca=N*H;
	float alpha=acos(ca);
	float ee=tan(alpha)/m_m;
	float D = exp(-ee*ee)/(M_PI*m_m*m_m*ca*ca*ca*ca);

	// Estimate G : masking
	float G=pFloat::min((2.0*(N*H)*(N*V)/(V*H)),(2.0*(N*H)*(N*L)/(V*H)));
	G=pFloat::min(1.0,G);

	// Estimate F : Fresnel coefficient
	float c=V*H;
	float n=(1+m_sFo)/(1-m_sFo);
	float g=sqrt(n*n+c*c-1);
	float gmc=g-c, gpc=g+c;
	float cgpc=c*gpc-1.0, cgmc=c*gmc+1.0;
	float F=0.5*(gmc*gmc)/(gpc*gpc)*(1.0+(cgpc*cgpc)/(cgmc*cgmc));

	float Rs = F*D*G / (4*(N*L)*(N*V));
	return (Rs+m_Kd/M_PI);
}


//=================================================================
void rCook::resetParam(int i)
{
	switch(i) {
		case 1:	m_Kd  = 0.0; break;
		case 2:	m_sFo = 0.0; break;
		case 3:	m_m   = 0.01; break;
	}
}


//=================================================================
void rCook::changeParam(int i, float val)
{
	float Fo;
	switch(i) {
		case 1:
			m_Kd+=val;
			break;
		case 2:
			Fo=m_sFo*m_sFo;
			Fo+=val;
			m_sFo=sqrt(Fo);
			break;
		case 3:
			m_m += val;
			break;
	}

	m_Kd=pFloat::min(m_Kd,1.0);
	m_Kd=pFloat::max(m_Kd,EPSILON);
	m_sFo=pFloat::max(m_sFo,0.0);
	m_sFo=pFloat::min(m_sFo,1.0f-EPSILON);
	m_m=pFloat::max(m_m,0.01);
}


//=================================================================
char *rCook::paramValues()
{
	sprintf(m_values,"CookT : Kd=%2.2f   Fo=%2.2f   m=%2.2f",
					m_Kd,m_sFo*m_sFo,m_m);
	return m_values;
}

//=================================================================
// OREN-NAYAR model
//=================================================================

//=================================================================
float rOrenNayar::fr(pVect rVi, pVect Vo, pVect N)
{
	// Theta and Phi angles
	pVect rVo = -Vo;
	float cto = N*rVo;
	float to = acos(cto);
	float cti = N*rVi;
	float ti = acos(cti);

	// Phi vectors in face plane
	pVect rVop = rVo-N*cto;
	//rVop.normalize();
	float normVop = rVop.norm();
	if(normVop<=EPSILON)
        rVop = INFTY_BRDF;
    else
        rVop /= normVop;
	pVect rVip = rVi-N*cti;
	//rVip.normalize();
	float normVip = rVip.norm();
	if(normVip<=EPSILON)
        rVip = INFTY_BRDF;
    else
        rVip /= normVip;

	float c_popi;
	if(rVip.x>=INFTY_BRDF && rVop.x>=INFTY_BRDF)
        c_popi = INFTY_BRDF;
    else
        c_popi = rVip*rVop;

	// max angles alpha and beta
	float alpha = pFloat::max(to,ti);
	float beta = pFloat::min(to,ti);

	// C1 value
	float sigma_sigma=m_sigma*m_sigma;
	float c1 = 1.0f-0.5f*(sigma_sigma)/(sigma_sigma+0.33f);

	// C2 value
	float c2, c2p = 2.0f*beta/M_PI;
	if(c_popi>0.0f)
		c2=0.45f*(sigma_sigma)/(sigma_sigma+0.09f)*sin(alpha);
	else
		c2=0.45f*(sigma_sigma)/(sigma_sigma+0.09f)*(sin(alpha)-c2p*c2p*c2p);

	// C3 value
	float c3=4.0f*beta*alpha/(M_PI*M_PI);
	c3 *= c3;
	c3*=0.125f*(sigma_sigma)/(sigma_sigma+0.09f);

	// L1 value
	float L1=m_Kd/M_PI*(c1+c2*c_popi*tan(beta)
            +(1.0f-pFloat::abs(c_popi))*c3*tan((alpha+beta)/2.0f));

	// L2 value
	float L2=0.17f*(m_Kd*m_Kd)/M_PI*(sigma_sigma)/(sigma_sigma+0.13f)
						*(1.0f-c_popi*c2p*c2p);

	// Resulting BRDF
	return L1+L2;
}


//=================================================================
void rOrenNayar::resetParam(int i)
{
	switch(i) {
		case 1: m_Kd    = 0.0; break;
		case 2:	m_sigma = 0.0; break;
	}
}


//=================================================================
void rOrenNayar::changeParam(int i, float val)
{
	switch(i) {
	case 1:
		m_Kd+=val;
		break;
	case 2:
		m_sigma += val;
		break;
	}

	m_Kd=pFloat::min(m_Kd,1.0);
	m_Kd=pFloat::max(m_Kd,EPSILON);

	m_sigma=pFloat::min(m_sigma,M_PI/2.0);
	m_sigma=pFloat::max(m_sigma,0.0);
}

//=================================================================
char *rOrenNayar::paramValues()
{
	sprintf(m_values,"OrenN : Kd=%2.2f   sigma=%2.2f", m_Kd,m_sigma);
	return m_values;
}










