//=================================================================
// Camera definition for OpenGL
//
// Author : Daniel Meneveaux
// Date   : July 2008
//=================================================================

#include "camera.hpp"
#include <sys/time.h>
#include <stdio.h>



//=================================================================
static void renderBitmapString(float x, float y, float height, void *font, char *string)
{
	char *c;
	glRasterPos2f(x, height-y);
	for (c=string; *c != '\0'; c++) glutBitmapCharacter(font, *c);
}


//=================================================================
camera::camera(ushort x, ushort y)
{
	
	m_fLambert = new rLambert();
	m_fPhong = new rPhong();
	m_fCookT = new rCook();
	m_fOrenN = new rOrenNayar();
	
	m_Fr = m_fPhong;
  m_oDist = m_Dist = 1.7;
  m_oRx = m_Rx = 0.0;
  m_oRy = m_Ry = -45.0;
	m_angle = 45;
  m_imgWidth=x;
  m_imgHeight=y;
	m_srcTheta=30;
	m_srcPhi = 180;
	m_srcPow = 1.0;
  m_seeFps = false;
	m_wireframe = 2;
	m_3Dview = false;
	m_cutPlaneAngle=0;
	m_cylPlaneAngle=30;
  m_noVue = 0;
  gpuInit();
}

//=================================================================
void camera::gpuInit()
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(m_angle, 1.0, 0.1, 10.0);
  glClearColor(1.0,1.0,1.0,1.0);
  glShadeModel(GL_SMOOTH);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);
	glEnable(GL_LIGHT0);
	GLfloat light_ambient[] = { 0.1, 0.1, 0.1, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
	GLfloat light_diffuse[] = { 0.3, 0.3, 0.3, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse); 
	glEnable(GL_LIGHT1);
	GLfloat light_ambient1[] = { 0.0, 0.0, 0.0, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_AMBIENT, light_ambient1); 
	GLfloat light_diffuse1[] = { 1.2, 1.2, 1.2, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_DIFFUSE, light_diffuse1); 

	
	glEnable(GL_LIGHTING);
	glMatrixMode(GL_MODELVIEW);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

}


//=================================================================
void camera::setImageSize(ushort x, ushort y)
{
  m_imgWidth = x;
  m_imgHeight = y;
}

//=================================================================
void camera::changeIncidentAngle(int val)
{
	m_srcTheta+=val;
	m_srcTheta=pFloat::max(m_srcTheta,0);
	m_srcTheta=pFloat::min(m_srcTheta,90);
}


//=================================================================
void camera::changeCutPlaneAngle(float val)
{
	m_cutPlaneAngle+=val;
	if(pFloat::abs(m_cutPlaneAngle)<0.5) m_cutPlaneAngle=0;
	m_cutPlaneAngle=pFloat::max(m_cutPlaneAngle,-90);
	m_cutPlaneAngle=pFloat::min(m_cutPlaneAngle,90);
}


//=================================================================
void camera::changeCylPlaneAngle(float val)
{
	m_cylPlaneAngle+=val;
	m_cylPlaneAngle=pFloat::min(m_cylPlaneAngle,90);
	m_cylPlaneAngle=pFloat::max(m_cylPlaneAngle,1);
}

//=================================================================
void camera::brdfParam(int id, float val)
{
	m_Fr->changeParam(id,val);
}


//=================================================================
void camera::changeModel()
{
	if(m_Fr==m_fLambert) m_Fr=m_fPhong;
	else if(m_Fr==m_fPhong) m_Fr=m_fCookT;
  else if(m_Fr==m_fCookT) m_Fr=m_fOrenN;
  else if(m_Fr==m_fOrenN) m_Fr=m_fLambert;
}




//=================================================================
void camera::gpuBeginView()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// camera position
	glLoadIdentity();
  glTranslated(-0.1,-0.15,0);
  glTranslated(0,0,-m_Dist);
  glRotatef(m_Ry, 1.0, 0.0, 0.0);
  glRotatef(m_Rx, 0.0, 0.0, 1.0);
}

//=================================================================
void camera::gpuBeginView2D()
{
	glDisable(GL_LIGHTING);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, m_imgWidth/2, 0, m_imgHeight);

  glMatrixMode(GL_MODELVIEW);
}

//=================================================================
void camera::gpuEndView2D()
{
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
	glEnable(GL_LIGHTING);
}

//=================================================================
void camera::gpuEndView()
{
  if(m_seeFps) displayFps();
  glutSwapBuffers();
}


//=================================================================
void camera::gpuLighting()
{
	pVect Vi,C(0,0,0);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta), pFloat::degRad(m_srcPhi),C,1.5);
	GLfloat light_position[] = { Vi.x, Vi.y, Vi.z }; 
	glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	GLfloat light_position1[] = { 0.0, 0.0, 5.0, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_POSITION, light_position1);
}

//=================================================================
void camera::gpuDrawScene()
{
  gpuBeginView();

	if(m_3Dview) {
		glViewport(0,0,(GLsizei)m_imgWidth/2.0,(GLsizei)m_imgHeight);
		gpuLighting();
		gpuDrawSquare();
		gpuDrawInOutDirections();
		//if(m_wireframe>0) gpuDrawBRDF(1);      // for uniform angle meshes
		//if(m_wireframe%2==0) gpuDrawBRDF(0);   // for uniform angle meshes
		if(m_wireframe>0) m_model3D.drawIcosaedre(m_Fr,pVect(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi)),m_srcPow,GL_POLYGON);
		if(m_wireframe%2==0) m_model3D.drawIcosaedre(m_Fr,pVect(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi)),m_srcPow,GL_LINE_LOOP);
	} else {
		glViewport(0,(GLsizei)m_imgHeight/2,(GLsizei)m_imgWidth/2.0,(GLsizei)m_imgHeight/2.0);
		gpuBeginView2D();
		gpuDrawBrdfCutPlane();
		gpuEndView2D();
	
		glViewport(0,0,(GLsizei)m_imgWidth/2.0,(GLsizei)m_imgHeight/2);
		gpuBeginView2D();
		gpuDrawBrdfCylinderPlane();
		gpuEndView2D();
	}

	
	glViewport(m_imgWidth/2.0,0,(GLsizei)m_imgWidth/2.0,(GLsizei)m_imgHeight);
	gpuBeginView2D();
	gpuDrawBrdfAreaDisk();
	gpuDrawInfo();
	gpuEndView2D();
	
	gpuEndView();
}



//=================================================================
void camera::gpuDrawSquare()
{
	GLfloat mat_diff[] = { 0.2, 0.2, 0.2, 1.0 }; 
	glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diff); 
	GLfloat mat_spec[] = { 0.0, 0.0, 0.0, 1.0 }; 
	glMaterialfv(GL_FRONT, GL_SPECULAR, mat_spec); 
	GLfloat mat_shin[] = { 0.0 }; 
	glMaterialfv(GL_FRONT, GL_SHININESS, mat_shin); 
	
	glBegin(GL_POLYGON);
		glNormal3f(0.0,0.0,1.0);
		glVertex3f(-0.3,-0.5,-0.01);
		glNormal3f(0.0,0.0,1.0);
		glVertex3f( 1.7,-0.5,-0.01);
		glNormal3f(0.0,0.0,1.0);
		glVertex3f( 1.7, 0.5,-0.01);
		glNormal3f(0.0,0.0,1.0);
		glVertex3f(-0.3, 0.5,-0.01);		
	glEnd();
}

//=================================================================
void camera::gpuDrawInOutDirections()
{
	glDisable(GL_LIGHTING);
	
	pVect Vi,C(0,0,0);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta), pFloat::degRad(m_srcPhi),C,1.0);
	
	glLineWidth(3.0);
	glColor3f(1,1,0);
	glBegin(GL_LINE_LOOP);
		glVertex3f(0.0,0.0,0.0);
		glVertex3f(Vi.x,Vi.y,Vi.z);
	glEnd();
	Vi.x=-Vi.x;
	glColor3f(1,0,0);
	glBegin(GL_LINE_LOOP);
		glVertex3f(0.0,0.0,0.0);
		glVertex3f(Vi.x,Vi.y,Vi.z);
	glEnd();
	glLineWidth(1.0);

	glEnable(GL_LIGHTING);
}

//=================================================================
// mode=0 => wireframe
// mode=1 => solid
void camera::gpuDrawBRDF(uchar mode)
{	
	int nbT=30, nbP=50;
	pVect P1,P2,P3,P4,Vi,C(0.0,0.0,0.0),N(0.0,0.0,1.0);
	GLint MODE=GL_POLYGON;
	
	float R1,R2,R3,R4;

	if(!mode) {
		glDisable(GL_LIGHTING);
		glColor3f(0.0,0.0,0.0);
		MODE=GL_LINE_LOOP;
	} else {
		glEnable(GL_BLEND);
		GLfloat mat_diff[] = { 0.7, 0.5, 0.2, 0.7 }; 
		glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diff); 
		GLfloat mat_spec[] = { 0.5, 0.5, 0.5, 1.0 }; 
		glMaterialfv(GL_FRONT, GL_SPECULAR, mat_spec);
		GLfloat mat_shin[] = { 100.0 }; 
		glMaterialfv(GL_FRONT, GL_SHININESS, mat_shin); 
	}
	
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi),C,1.0);
	
	float dt = M_PI/(2.0*nbT);
	float dp = 2.0*M_PI/nbP;
	
	for(int i=0; i<nbT; i++) {
		for(int j=0; j<nbP; j++) {

			P1.setSphereVertex(i*dt,j*dp,C,1.0);         // bottom-left
			P2.setSphereVertex((i+1)*dt,j*dp,C,1.0);     // bottom-right
			P3.setSphereVertex((i+1)*dt,(j+1)*dp,C,1.0); // top-right
			P4.setSphereVertex(i*dt,(j+1)*dp,C,1.0);     // top-left

			R1 = m_Fr->fr(Vi,-P1,N)*m_srcPow;
			R2 = m_Fr->fr(Vi,-P2,N)*m_srcPow;
			R3 = m_Fr->fr(Vi,-P3,N)*m_srcPow;
			R4 = m_Fr->fr(Vi,-P4,N)*m_srcPow;

			if(!mode) {
				R1+=0.001;
				R2+=0.001;
				R3+=0.001;
				R4+=0.001;
			}

			glBegin(MODE);
				glNormal3f(P1.x,P1.y,P1.z);
				P1*=R1;
				glVertex3f(P1.x,P1.y,P1.z);
				glNormal3f(P2.x,P2.y,P2.z);
				P2*=R2;
				glVertex3f(P2.x,P2.y,P2.z);
				glNormal3f(P3.x,P3.y,P3.z);
				P3*=R3;
				glVertex3f(P3.x,P3.y,P3.z);
				glNormal3f(P4.x,P4.y,P4.z);
				P4*=R4;
				glVertex3f(P4.x,P4.y,P4.z);
			glEnd();
		}
	}
	
	if(mode) glEnable(GL_LIGHTING);
	else glDisable(GL_BLEND);
}


//=================================================================
void camera::gpuDrawCircle(int xc,int yc,int r)
{
	int nbE=100;
	float x,y,t,dt=M_PI*2.0/nbE;
	
	glBegin(GL_LINE_LOOP);
	for(int i=0; i<nbE; i++) {
		t=i*dt;
		x=xc+r*cos(t);
		y=yc+r*sin(t);
		glVertex2f(x,y);
	}
	glEnd();
}

//=================================================================
void camera::gpuDrawBrdfSplatDisk()
{
	char TXT[50];
	int sx = m_imgWidth/2, sy = m_imgHeight; // viewport screen size
	int cx = sx/2, cy = sy/2;                // viewport screen center
	int pixSize = 3;
	float rmin=2.0, rmax=0.0;
	
	// display separation line between viewports
	glColor3f(0,0,0);
	glBegin(GL_LINE_LOOP);
		glVertex2f(1.0,0.0);
		glVertex2f(1.0,sy);
	glEnd();

	// Incident direction
	pVect Vi,N(0,0,1),C(0,0,0);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi),C,1.0);

	// draw circle
	glPointSize(pixSize);
	glBegin(GL_POINTS);
	for(int j=0; j<sy/pixSize; j++) {
		for(int i=0; i<sx/pixSize; i++) {
		
			float xc=i*pixSize+pixSize/2;      // x absolute pixel center
			float yc=j*pixSize+pixSize/2;      // y absolute pixel center
						
			float xd=(float)(xc-cx)/(sx*0.3);  // x coord in disk system
			float yd=(float)(yc-cy)/(sy*0.3);  // y coord in disk system
			float d = sqrt(xd*xd+yd*yd);
				
			if(d<=1.0) {
				float zd = sqrt(1.0-d*d);
				pVect Vo(xd,yd,zd);
				float r = m_Fr->fr(Vi,-Vo,N)*m_srcPow;
				if(r<rmin) rmin=r;
				if(r>rmax) rmax=r; 
				glColor3f(r,r,r);
				glVertex2f(xc,yc);
			}
		}
	}
	glEnd();
	
	// Display values rectangle
	glBegin(GL_POLYGON);
		glColor3f(rmin,rmin,rmin);
		glVertex2f(70.0,sy-10.0);
		glVertex2f(70.0,sy-30.0);
		glColor3f(rmax,rmax,rmax);
		glVertex2f(sx-70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-10.0);
	glEnd();

	// Display black rectangle border
	glBegin(GL_LINE_LOOP);
		glColor3f(0.0,0.0,0.0);
		glVertex2f(70.0,sy-9.0);
		glVertex2f(70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-10.0);
	glEnd();
	
	// Display values rectangle text
	sprintf(TXT,"min=%2.2f",rmin);
  renderBitmapString(70,10,m_imgHeight-30,GLUT_BITMAP_HELVETICA_10,TXT);
	sprintf(TXT,"max=%2.2f",rmax);
  renderBitmapString(m_imgWidth/2-120,10,m_imgHeight-30,GLUT_BITMAP_HELVETICA_10,TXT);

	// Display circle values
	float rc=sx*0.3;
	glColor3f(0.6,0.6,0.6);
	sprintf(TXT,"0",rmin);
  renderBitmapString(cx+2,10,256,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,rc*sin(pFloat::degRad(20.0)));
	sprintf(TXT,"20",rmin);
  renderBitmapString(cx+2,10,204,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,rc*sin(pFloat::degRad(45.0)));
	sprintf(TXT,"45",rmin);
  renderBitmapString(cx+2,10,148,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,rc*sin(pFloat::degRad(60.0)));
	sprintf(TXT,"60",rmin);
  renderBitmapString(cx+2,10,123,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,rc);
	sprintf(TXT,"90",rmin);
  renderBitmapString(cx+2,10,103,GLUT_BITMAP_HELVETICA_10,TXT);
	
	// draw circle coord system
	glColor3f(0.0,0.0,0.0);
	glBegin(GL_LINE_LOOP);
		glVertex2f(cx,80.0);
		glVertex2f(cx,430.0);
	glEnd();
	glBegin(GL_LINE_LOOP);
		glVertex2f(80.0,cy);
		glVertex2f(430.0,cy);
	glEnd();

	// incident and mirror direction on circle
	glPointSize(5.0);
	glBegin(GL_POINTS);
		glColor3f(1,0,0);
		glVertex2f(Vi.x*sx*0.3+cx, Vi.y*sx*0.3+cy);
		glColor3f(0,0,1);
		glVertex2f(-Vi.x*sx*0.3+cx, -Vi.y*sx*0.3+cy);
	glEnd();

}


//=================================================================
void camera::gpuDrawBrdfAreaDisk()
{
	char TXT[50];
	int sx = m_imgWidth/2, sy = m_imgHeight; // viewport screen size
	int cx = sx/2, cy = sy/2;                // viewport screen center
	int pixSize = 4;
	float rmin=2.0, rmax=0.0;
	
	// display separation line between viewports
	glColor3f(0,0,0);
	glBegin(GL_LINE_LOOP);
		glVertex2f(1.0,0.0);
		glVertex2f(1.0,sy);
	glEnd();

	// Incident direction
	pVect Vi,Vo,N(0,0,1),C(0,0,0);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi),C,1.0);

	// draw circle
	glPointSize(pixSize);
	glBegin(GL_POINTS);
	for(int j=0; j<sy/pixSize; j++) {
		for(int i=0; i<sx/pixSize; i++) {
		
			float xc=i*pixSize+pixSize/2.0;      // x absolute pixel center
			float yc=j*pixSize+pixSize/2.0;      // y absolute pixel center
			
			float xd=(float)(xc-cx)/(sx*0.25);  // x coord in disk system
			float yd=(float)(yc-cy)/(sy*0.25);  // y coord in disk system
			float d = sqrt(xd*xd+yd*yd);

			if(d*d<=2.0) {
				float th=2.0*asin(d/2.0);
				float phi=acos(xd/d);
				Vo.setSphereVertex(th,phi,C,1.0);
				float r = m_Fr->fr(Vi,-Vo,N)*m_srcPow;
				if(r<rmin) rmin=r;
				if(r>rmax) rmax=r; 
				glColor3f(r,r,r);
				glVertex2f(xc,yc);
			}
		}
	}
	glEnd();
	
	// Display values rectangle
	glBegin(GL_POLYGON);
		glColor3f(0.0f,0.0f,0.0f);
		glVertex2f(70.0,sy-10.0);
		glVertex2f(70.0,sy-30.0);
		glColor3f(rmax,rmax,rmax);
		glVertex2f(sx-70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-10.0);
	glEnd();

	// Display black rectangle border
	glBegin(GL_LINE_LOOP);
		glColor3f(0.0,0.0,0.0);
		glVertex2f(70.0,sy-9.0);
		glVertex2f(70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-30.0);
		glVertex2f(sx-70.0,sy-10.0);
	glEnd();
	
	sprintf(TXT,"min=%2.2f",0.0f);
  renderBitmapString(70,10,m_imgHeight-30,GLUT_BITMAP_HELVETICA_10,TXT);
	sprintf(TXT,"max=%2.2f",rmax);
  renderBitmapString(m_imgWidth/2-120,10,m_imgHeight-30,GLUT_BITMAP_HELVETICA_10,TXT);

	float rc=sx*0.25;
	glColor3f(1.0,1.0,1.0);
	sprintf(TXT,"0",rmin);
  renderBitmapString(cx+2,10,256,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,2.0*rc*sin(pFloat::degRad(20.0)/2.0));
	sprintf(TXT,"20",rmin);
  renderBitmapString(cx+2,10,212,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,2.0*rc*sin(pFloat::degRad(45.0)/2.0));
	sprintf(TXT,"45",rmin);
  renderBitmapString(cx+2,10,159,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,2.0*rc*sin(pFloat::degRad(60.0)/2.0));
	sprintf(TXT,"60",rmin);
  renderBitmapString(cx+2,10,128,GLUT_BITMAP_HELVETICA_10,TXT);
	gpuDrawCircle(cx,cy,rc);
	sprintf(TXT,"90",rmin);
  renderBitmapString(cx+2,10,90,GLUT_BITMAP_HELVETICA_10,TXT);
	glColor3f(1.0,1.0,1.0);
	glLineWidth(5.0);
	gpuDrawCircle(cx,cy,rc*sqrt(2.0));
	gpuDrawCircle(cx,cy,rc*sqrt(2.0)+2.0);
	glLineWidth(1.0);
	
	// draw circle coord system
	glColor3f(1.0,1.0,1.0);
	glBegin(GL_LINE_LOOP);
		glVertex2f(cx,60.0);
		glVertex2f(cx,450.0);
	glEnd();
	glBegin(GL_LINE_LOOP);
		glVertex2f(60.0,cy);
		glVertex2f(450.0,cy);
	glEnd();

	// incident and mirror direction on circle
	glPointSize(5.0);
	glBegin(GL_POINTS);
		glColor3f(1,0,0);
		glVertex2f(2.0*rc*sin(pFloat::degRad(m_srcTheta)/2.0)+cx,cy);
		glColor3f(1,1,0);
		glVertex2f(-2.0*rc*sin(pFloat::degRad(m_srcTheta)/2.0)+cx,cy);
	glEnd();
	
	// cut plane
	float begx=cos(pFloat::degRad(m_cutPlaneAngle))*sx*0.4;
	float begy=sin(pFloat::degRad(m_cutPlaneAngle))*sx*0.4;
	glColor3f(0,0,1);
	glBegin(GL_LINE_STRIP);
		glVertex2f(begx+cx,begy+cy);
		glVertex2f(-begx+cx,-begy+cy);
	glEnd();
	sprintf(TXT,"%2.0f",m_cutPlaneAngle);
	renderBitmapString(begx+cx+3,10,begy+cy+10,GLUT_BITMAP_HELVETICA_10,TXT);

	// cylinder plane
	glColor3f(0,0.7,0);
	gpuDrawCircle(cx,cy,2.0*rc*sin(pFloat::degRad(m_cylPlaneAngle)/2.0));
	sprintf(TXT,"%2.0f",m_cylPlaneAngle);
	renderBitmapString(cx+2,10,cy-20,GLUT_BITMAP_HELVETICA_10,TXT);
}

//=================================================================
void camera::gpuDrawInfo()
{
	glColor3f(0.3,0.0,0.3);
	char TXT[50];
	sprintf(TXT,"Incident angle : %2.0f",m_srcTheta);
  renderBitmapString(10,10,35,GLUT_BITMAP_HELVETICA_10,TXT);
	sprintf(TXT,"BRDF scale : %2.2f",m_srcPow);
  renderBitmapString(10,10,20,GLUT_BITMAP_HELVETICA_10,TXT);
	renderBitmapString(200,10,20,GLUT_BITMAP_HELVETICA_10,m_Fr->paramValues());
}

//=================================================================
void camera::gpuDrawBrdfCutPlane()
{
	float lg=m_imgWidth/2-40;
	float ht=m_imgHeight-50;
	float l=25;
	float b=29;
	char TXT[10];

	// display separation line between viewports
	glColor3f(0,0,0);
	glBegin(GL_LINE_LOOP);
		glVertex2f(1.0,1.0);
		glVertex2f(m_imgWidth/2,1.0);
	glEnd();

	// drawing rectangle Area
	glColor3f(0.8,0.8,0.8);
	glBegin(GL_POLYGON);
		glVertex2f(l,b);
		glVertex2f(l+lg,b);
		glVertex2f(l+lg,b+ht);
		glVertex2f(l,b+ht);
	glEnd();

	// display Ox axis
	glColor3f(0,0,0);
	glBegin(GL_LINES);
	glVertex2f(l-2,b);
	glVertex2f(l+lg+2,b);
	for(int i=-90; i<=90; i+= 10) {
		glVertex2f(l+(i+90)*lg/180,b-2);
		glVertex2f(l+(i+90)*lg/180,b+4);
	}
	glEnd();
	renderBitmapString(l+lg-20,0,b+7,GLUT_BITMAP_HELVETICA_10,"deg.");
	for(int i=-90; i<=90; i+= 10) {
		sprintf(TXT,"%2d",i);
		int dec=(i<0)?-3:5;
		renderBitmapString(dec+l+(i+90)*lg/180-10,13,20,GLUT_BITMAP_HELVETICA_10,TXT);
	}

	// Compute BRDF values and estimate max
	float rMax=1.0, R, w=pFloat::degRad(m_cutPlaneAngle);
	pVect Vi,Vo,C(0,0,0),N(0,0,1);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi),C,1.0);
	static float *tab=new float[100];
	for(int vo=-90; vo<=90; vo+= 2) {
		float vvo=pFloat::degRad(vo);
		Vo.setSphereVertex(vvo,pFloat::degRad(m_srcPhi+180),C,1.0);
		Vo.rotateVectorSphere(N,pVect(cos(w),sin(w),0),pVect(cos(w+M_PI/2),sin(w+M_PI/2),0));
		R = m_Fr->fr(Vi,-Vo,N)*m_srcPow;
		tab[(vo+90)/2]=R;
		rMax=pFloat::max(rMax,R);
	}
	rMax*=1.03;

	// display Oy axis
	glColor3f(0,0,0);
	glBegin(GL_LINES);
	glVertex2f(l,b-2);
	glVertex2f(l,b+ht+2);
	for(float i=0.1; i<=1.1; i+= 0.1) {
		glVertex2f(l-2,b+i*ht);
		glVertex2f(l+2,b+i*ht);
	}
	glEnd();
	renderBitmapString(l+2,20,b+ht,GLUT_BITMAP_HELVETICA_10,"1/sr");
	for(float i=0.1; i<=1.1; i+= 0.1) {
		sprintf(TXT,"%.1f",i*rMax);
		renderBitmapString(l-20,7,b+i*ht,GLUT_BITMAP_HELVETICA_10,TXT);
	}

	// display intersection planes
	glColor3f(0.0,0.7,0.0);
	glBegin(GL_LINE_STRIP);
		glVertex2f(l+lg/2+m_cylPlaneAngle*lg/180,b);
		glVertex2f(l+lg/2+m_cylPlaneAngle*lg/180,b+ht);
	glEnd();
	glBegin(GL_LINE_STRIP);
		glVertex2f(l+lg/2-m_cylPlaneAngle*lg/180,b);
		glVertex2f(l+lg/2-m_cylPlaneAngle*lg/180,b+ht);
	glEnd();
	
	// display BRDF
	glColor3f(0.0,0.0,1.0);
	glBegin(GL_LINE_STRIP);
	for(int vo=-90; vo<=90; vo+= 2) {
		glVertex2f((vo+90.0)*lg/180.0+l,tab[(vo+90)/2]*ht/rMax+b);
	}
	glEnd();
}


//=================================================================
void camera::gpuDrawBrdfCylinderPlane()
{
	float lg=m_imgWidth/2-40;
	float ht=m_imgHeight-50;
	float l=25;
	float b=29;
	char TXT[10];

	// drawing rectangle Area
	glColor3f(0.8,0.8,0.8);
	glBegin(GL_POLYGON);
		glVertex2f(l,b);
		glVertex2f(l+lg,b);
		glVertex2f(l+lg,b+ht);
		glVertex2f(l,b+ht);
	glEnd();
	
	// display Ox axis
	glColor3f(0,0,0);
	glBegin(GL_LINES);
	glVertex2f(l-2,b);
	glVertex2f(l+lg+2,b);
	for(int i=-180; i<=180; i+= 30) {
		glVertex2f(l+(i+180)*lg/360,b-2);
		glVertex2f(l+(i+180)*lg/360,b+4);
	}
	glEnd();
	renderBitmapString(l+lg-20,0,b+7,GLUT_BITMAP_HELVETICA_10,"deg.");
	for(int i=-180; i<=180; i+= 30) {
		sprintf(TXT,"%2d",i);
		int dec=(i<=90)?3:0;
		dec=(i<0)?-3:dec;
		renderBitmapString(dec+l+(i+180)*lg/360-10,13,20,GLUT_BITMAP_HELVETICA_10,TXT);
	}

	// compute BRDF max
	float R, rMax = 1.0;
	static float *tab=new float[370];
	pVect Vi,Vo,C(0,0,0),N(0,0,1);
	Vi.setSphereVertex(pFloat::degRad(m_srcTheta),pFloat::degRad(m_srcPhi),C,1.0);
	for(int ph=0; ph<360; ph++) {
		Vo.setSphereVertex(pFloat::degRad(m_cylPlaneAngle),pFloat::degRad(ph+180),C,1.0);
		R = m_Fr->fr(Vi,-Vo,N)*m_srcPow;
		tab[ph]=R;
		rMax=pFloat::max(rMax,R);
	}
	rMax*=1.05;


	// display Oy axis
	glColor3f(0,0,0);
	glBegin(GL_LINES);
	glVertex2f(l,b-2);
	glVertex2f(l,b+ht+2);
	for(float i=0.1; i<=1.1; i+= 0.1) {
		glVertex2f(l-2,b+i*ht);
		glVertex2f(l+2,b+i*ht);
	}
	glEnd();
	renderBitmapString(l+2,20,b+ht,GLUT_BITMAP_HELVETICA_10,"1/sr");
	for(float i=0.1; i<=1.1; i+= 0.1) {
		sprintf(TXT,"%.1f",i*rMax);
		renderBitmapString(l-20,7,b+i*ht,GLUT_BITMAP_HELVETICA_10,TXT);
	}

	// display intersection planes
	glColor3f(0.0,0.0,1.0);
	int ag=(m_cutPlaneAngle<0)?m_cutPlaneAngle+180:m_cutPlaneAngle;
	glBegin(GL_LINE_STRIP);
		glVertex2f(l+ag*lg/360,b);
		glVertex2f(l+ag*lg/360,b+ht);
	glEnd();
	glBegin(GL_LINE_STRIP);
		glVertex2f(l+(ag+180)*lg/360,b);
		glVertex2f(l+(ag+180)*lg/360,b+ht);
	glEnd();	
		
	// display BRDF
	glColor3f(0.0,0.7,0.0);
	glBegin(GL_LINE_STRIP);
	for(int ph=0; ph<360; ph++) {
		glVertex2f(l+ph*lg/360,tab[ph]*ht/rMax+b);
	}
	glEnd();
}

//=================================================================
// MOUSE MANAGEMENT
//=================================================================

//=================================================================
void camera::mouseClick(int x,int y, uchar mode)
{
  m_modeMouse = mode;
  switch (m_modeMouse) {
  case TRACKBALL_MODE:
    mouseClickRotation(x,y);
    break;
  case DISTANCE_MODE:
    mouseClickDistance(x,y);
    break;
  case SRC_MOVE_MODE:
    mouseClickSrcMove(x,y);
    break;
	case SRC_POW_MODE:
    mouseClickSrcPow(x,y);
    break;
  }
}


//=================================================================
void camera::mouseMove(int x,int y)
{
  switch (m_modeMouse) {
  case TRACKBALL_MODE:
    mouseMoveRotation(x,y);
    break;
  case DISTANCE_MODE:
    mouseMoveDistance(x,y);
    break;
  case SRC_MOVE_MODE:
    mouseMoveSrcMove(x,y);
    break;
  case SRC_POW_MODE:
    mouseMoveSrcPow(x,y);
    break;  }
}


//=================================================================
void camera::mouseRelease(int x,int y)
{
  switch (m_modeMouse) {
  case TRACKBALL_MODE:
    mouseReleaseRotation(x,y);
    break;
  case DISTANCE_MODE:
    mouseReleaseDistance(x,y);
    break;
  case SRC_MOVE_MODE:
    mouseReleaseSrcMove(x,y);
		break;
  case SRC_POW_MODE:
    mouseReleaseSrcPow(x,y);
		break;
	}
	m_modeMouse = NO_MODE;
}


//=================================================================
void camera::mouseClickRotation(int x, int y)
{
  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseMoveRotation(int x, int y)
{
  int DX = (x-m_oX);
  int DY = (y-m_oY);

  m_Rx = m_oRx+DX/2.0;
  m_Ry = m_oRy+DY/2.0;
  m_Ry = (m_Ry < -1) ? m_Ry : -1;
  m_Ry = (m_Ry > -150) ? m_Ry : -150;
}


//=================================================================
void camera::mouseReleaseRotation(int x, int y)
{
  m_oRx = m_Rx;
  m_oRy = m_Ry;
}


//=================================================================
void camera::mouseClickDistance(int x, int y)
{
  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseMoveDistance(int x, int y)
{
  int DY = (y-m_oY);
  m_Dist -= DY/20.0;
  if (m_Dist < 0.5) m_Dist = 0.5;
	if (m_Dist > 10.0) m_Dist = 10.0;
  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseReleaseDistance(int x, int y)
{
  m_oRx = m_Rx;
  m_oRy = m_Ry;
  m_oDist = m_Dist;
}




//=================================================================
void camera::mouseClickSrcMove(int x, int y)
{
  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseMoveSrcMove(int x, int y)
{
	int DY=(y-m_oY);
	m_srcTheta+=DY/5.0;
	m_srcTheta = pFloat::max(m_srcTheta,EPSILON);
	m_srcTheta = pFloat::min(m_srcTheta,89);

  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseReleaseSrcMove(int x, int y)
{
}




//=================================================================
void camera::mouseClickSrcPow(int x, int y)
{
  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseMoveSrcPow(int x, int y)
{
	int DY=(y-m_oY);
	m_srcPow-=DY/50.0;

	m_srcPow = pFloat::max(m_srcPow,1.0);

  m_oX=x;
  m_oY=y;
}


//=================================================================
void camera::mouseReleaseSrcPow(int x, int y)
{
}



//=================================================================
// DISPLAY OPTIONS
//=================================================================




//=================================================================
// time display
//
void timerDisplay(long nbs, long nbms, bool dms=true)
{
  char aff[100];
  long nbm, nbh;
  if(nbms < 0) {
   nbs--;
   nbms+=1000000;
  }
  nbms /= 1000;
  nbm = nbs/60;
  nbs -= nbm*60;
  nbh = nbm/60;
  nbm -= nbh*60;
  if(dms) sprintf(aff,"(%.2dh%.2d'%.2d\"%.3dms)",(int)nbh,(int)nbm,(int)nbs,(int)nbms);
  else sprintf(aff,"(%.2dh%.2d'%.2d\")",(int)nbh,(int)nbm,(int)nbs);
  cerr << aff;
}

//=================================================================
// Timer for estimating computing time.
// value of state :
//    - 0 => reinit
//    - 1 => memorize time from init point and display time
//
void timer(int state, float pct)
{
  static struct timeval start, endt;
  struct timezone tz;
  if(pct<0.0001) pct = 0.0001;

  switch (state) {
  case 0 :
    gettimeofday(&start,&tz);
    break;
  case 1:
    gettimeofday(&endt,&tz);
    long nbms, nbs, nbsr, nbst;
    nbms = endt.tv_usec - start.tv_usec;
    nbs = endt.tv_sec - start.tv_sec;
    timerDisplay(nbs,nbms);
    nbst = nbs/pct; //(pct*sqrt(pct));
    nbsr = nbst-nbs;
    cout << " total:";
    timerDisplay(nbst,0.0,false);
    break;
  }
}



//=================================================================
void camera::toggleSeeFps()
{
  extern void idle();
  m_seeFps=!m_seeFps;
  if(m_seeFps) glutIdleFunc(idle);
  else glutIdleFunc(NULL);
}

//=================================================================
void camera::toggleLeftView()
{
	m_3Dview=!m_3Dview;
}



//=================================================================
void camera::displayFps()
{
	static unsigned int frame=0;
	static int timebase=glutGet(GLUT_ELAPSED_TIME)-1;
	static char sTime[128];

  glColor3f(0.0f,0.0f,0.0f);

	glViewport(0,0,(GLsizei)m_imgWidth,(GLsizei)m_imgHeight);

  frame++;
  int elapsedTime=glutGet(GLUT_ELAPSED_TIME);
  if (elapsedTime - timebase > 1000)
  {
    sprintf(sTime,"fps: %4.2f",frame*1000.0/(elapsedTime-timebase));
    timebase = elapsedTime;
    frame = 0;
  }

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, m_imgWidth, 0, m_imgHeight);
  glMatrixMode(GL_MODELVIEW);
  renderBitmapString(m_imgWidth/2+20,10,m_imgHeight-50,GLUT_BITMAP_HELVETICA_10,sTime);
	glRasterPos2d(0,0);

  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
}
//=================================================================


//=================================================================
// DISPLAY OPTIONS
//=================================================================

void camera::toggleWireframe()
{
	m_wireframe = (m_wireframe+1)%3;
}

