// ================================================================
// Points, vectors and matrices operations
// Author : Daniel Meneveaux
// Date : Feb 2008
// ================================================================

#include <stdlib.h>
#include "point.hpp"


// ================================================================
// pFloat operations
// ================================================================
bool pFloat::equiv(const float a, const float b, const float c, const float ratio)
{	
	if(a/b<ratio) return false;
	if(b/a<ratio) return false;
	if(a/c<ratio) return false;
	if(c/a<ratio) return false;
	if(b/c<ratio) return false;
	if(c/b<ratio) return false;
	return true;
}




// ================================================================
// pVect operations
// ================================================================
void pVect::updateMinBox(const pVect &P)
{
  x=(P.x-SMALL<x)?P.x-SMALL:x;
  y=(P.y-SMALL<y)?P.y-SMALL:y;
  z=(P.z-SMALL<z)?P.z-SMALL:z;
}

// ================================================================
// update bounding box max-value given a 3D point
void pVect::updateMaxBox(const pVect &P)
{
  x=(P.x+SMALL>x)?P.x+SMALL:x;
  y=(P.y+SMALL>y)?P.y+SMALL:y;
  z=(P.z+SMALL>z)?P.z+SMALL:z;
}

// ================================================================
// "this" coords should be < v coords.
// If not, the methods reverses the corresponding values
//
void pVect::sortCoordsMinMax(pVect &vMin, pVect &vMax) const
{
	float aux;
	if(x<0) {
		aux=vMin.x;
		vMin.x=vMax.x;
		vMax.x=aux;
	}

	if(y<0) {
		aux=vMin.y;
		vMin.y=vMax.y;
		vMax.y=aux;
	}

	if(z<0) {
		aux=vMin.z;
		vMin.z=vMax.z;
		vMax.z=aux;
	}

}


// ================================================================
// Returns the triangle area
// (basis*height) / 2
// V1=(P1,P2) and V2=(P1,P3)
// the height is estimated from the projection of V2 on V1 (dot product)
//
float pVect::getTriangleArea(const pVect &P1,const pVect &P2,const pVect &P3)
{
  pVect V1, V2, Ph=P1, VF;
  float nv1c, ps, basis, height;
  V1=P2-P1;
	nv1c=V1.sqrNorm();
	if(nv1c<EPSILON) return 0.0;
  basis = sqrt(nv1c);
  V2=P3-P1;
  ps=V1*V2;
  ps = ps/nv1c;
	Ph.addX(ps,V1);
	VF=P3-Ph;
  height=VF.norm();
  return (basis*height/2.0);
}


// ================================================================
// construct u,v basis vectors from *this (normal)
//
void pVect::setBasisVectors(pVect &u, pVect &v) const
{
  pVect N = (*this);
  pVect rep(0,0,1);
  if(rep*N > 0.9) rep.set(1,0,0);
  u=rep^N; u.normalize();
  v=N^u;   v.normalize();
}

// ================================================================
void pVect::rotateVectorSphere(const pVect &n, const pVect &u, const pVect &v)
{
  pVect dir = *this;
	x=(u.x*dir.x)+(v.x*dir.y)+(n.x*dir.z);
	y=(u.y*dir.x)+(v.y*dir.y)+(n.y*dir.z);
	z=(u.z*dir.x)+(v.z*dir.y)+(n.z*dir.z);
}

// ================================================================
// creates a point on a sphere, given theta/phi/r
//
void pVect::setSphereVertex(float th, float ph, const pVect &C, float r)
{
	x=C.x+r*sin(th)*cos(ph);
	y=C.y+r*sin(th)*sin(ph);
	z=C.z+r*cos(th);
}

// ================================================================
// Matrix operations
// ================================================================



// ================================================================
matrix::matrix(const pVect &l1, const pVect &l2, const pVect &l3)
{
  L1=l1;
  L2=l2;
  L3=l3;
};

// ================================================================
void matrix::setRotation(const pVect &u, const pVect &v, const pVect &w)
{
  L1.set(u.x,v.x,w.x);
  L2.set(u.y,v.y,w.y);
  L3.set(u.z,v.z,w.z);
};

// ================================================================
void matrix::setRotationX(float A)
{
  L1.set(1,0,0);
  L2.set(0,cos(A),-sin(A));
  L3.set(0,sin(A),cos(A));
};

// ================================================================
void matrix::setRotationY(float A)
{
  L1.set(cos(A),0,-sin(A));
  L2.set(0,1,0);
  L3.set(sin(A),0,cos(A));
};

// ================================================================
void matrix::setRotationZ(float A)
{
  L1.set(cos(A),-sin(A),0);
  L2.set(sin(A),cos(A),0);
  L3.set(0,0,1);
};

// ================================================================
void matrix::transpose()
{
  pVect C1(L1.x,L2.x,L3.x);
  pVect C2(L1.y,L2.y,L3.y);
  pVect C3(L1.z,L2.z,L3.z);
  L1=C1;
  L2=C2;
  L3=C3;
};

// ================================================================
matrix matrix::operator*(const matrix &M)
{
  matrix T=M, R;
  R.L1.set(L1*M.L1,L1*M.L2,L1*M.L3);
  R.L2.set(L2*M.L1,L2*M.L2,L2*M.L3);
  R.L3.set(L3*M.L1,L3*M.L2,L3*M.L3);
  return R;
}

// ================================================================
pVect matrix::operator*(const pVect &V)
{
  pVect R(L1*V,L2*V,L3*V);
  return R;
}













