//=================================================================
// Operations used for points and vectors
//
// Author : Daniel Meneveaux
// Date   : May 2008
//=================================================================


#ifndef _point_hpp_
#define _point_hpp_

#include <math.h>
#include "types.hpp"


//=================================================================
// Tools for float
//
class pFloat {
public:
  inline static float abs(float x) { return ((x>=0.0)?x:-x); }
  inline static bool zero(float x) { return ((abs(x)<EPSILON)?true:false); }
  inline static float degRad(float D) { return (((float)(D) * (float)M_PI) / (float)180.0); }
  inline static float radDeg(float R) { return (((float)(R) * (float)180.0) / (float)M_PI); }
	inline static float min(float x, float val) { return (x<val)?x:val; };
	inline static float max(float x, float val) { return (x>val)?x:val; };
	inline static float min(float a, float b, float c) { return (a<b)?((a<c)?a:c):((b<c)?b:c); };	
	inline static float max(float a, float b, float c) { return (a>b)?((a>c)?a:c):((b>c)?b:c); };	
	static bool equiv(float a, float b, float c, float ratio);
};




//=================================================================
// Point and vector operations
//
class pVect {

public:
	float x,y,z;			// coordinates

public:

	// construction and default initialization
	pVect()
	: x(0), y(0), z(0) {};
	
	pVect(float th, float phi)
	: x(cos(phi)*sin(th)), y(sin(phi)*sin(th)), z(cos(th)) {};

	// constructs and initializes a pVect
	pVect(const float vx, const float vy, const float vz)
	:x(vx), y(vy), z(vz) {};

	// constructs and initializes a pVect from another pVect
	pVect(const pVect &v)
	: x(v.x), y(v.y), z(v.z) {};

	// end of life...
	~pVect() {}

  // stream operator
  friend ostream &operator<<(ostream &str, const pVect p)
  { str << "pVect : " << p.x << " " << p.y << " " << p.z; return str; }

	// sets pVect coordinates
	void set(const float vx, const float vy, const float vz)
	{x=vx;y=vy;z=vz;};

    // gets pVect coordinates
	void get(float &vx, float &vy, float &vz)
	{vx=x;vy=y;vz=z;};

    // copy
	void operator=(const float v)
	{x=v; y=v; z=v;};

	// dot product
	float operator*(const pVect &p) const
	{return (x*p.x+y*p.y+z*p.z);};

	// product with a scalar
	pVect operator*(const float v) const
	{return pVect(v*x,v*y,v*z);}

	friend pVect operator*(const float v, const pVect &p)
	{ return p*v;}

	// divide by a scalar
	pVect operator/(float v) const
	{return pVect(x/v,y/v,z/v);}

	// sums 2 vectors
	pVect operator+(const pVect &p) const
	{ return pVect(x+p.x,y+p.y,z+p.z);};

	// substracts 2 vectors
	pVect operator-(const pVect &p) const
	{ pVect res; res.set(x-p.x,y-p.y,z-p.z); return res;};

	// substract a vector
	void operator-=(const pVect &v)
	{x-=v.x; y-=v.y; z-=v.z;};

  // negate vector
  friend pVect operator-(pVect &V)
  {return pVect(-V.x,-V.y,-V.z);};

	// cross product
	pVect operator^(const pVect &p) const
	{pVect res(y*p.z-z*p.y,z*p.x-x*p.z,x*p.y-y*p.x); return res;};

	// add a scalar
	void operator+=(const float v)
	{x+=v; y+=v; z+=v;};

	// add a scalar
	void operator-=(const float v)
	{x-=v; y-=v; z-=v;};

	// add a vector
	void operator+=(const pVect &V)
	{x+=V.x; y+=V.y; z+=V.z;};

	// auto-multiply by a scalar
	void operator*=(const float v)
	{x*=v; y*=v; z*=v;};

	// auto-multiply by a vector (each coordinate)
	void operator*=(const pVect &v)
	{x*=v.x; y*=v.y; z*=v.z;};

	// auto-divide by a scalar
	void operator/=(const float v)
	{x/=v; y/=v; z/=v;};

	// auto-devide by a vector (each coordinate)
	void operator/=(const pVect &v)
	{x=(pFloat::zero(v.x))?INFINITY:x/=v.x; y=(pFloat::zero(v.y))?INFINITY:y/=v.y; z=(pFloat::zero(v.z))?INFINITY:z/=v.z; };

	// square norm
	float sqrNorm() const
	{return x*x+y*y+z*z;};

	// norm
	float norm() const
	{return sqrt(sqrNorm());};

	// normalize a vector
	void normalize()
	{float s=norm(); (*this)/=s;}

	// constructs a vector from two points
	void vectorize(const pVect &P1, const pVect &P2)
	{*this=P2-P1;};

	// adds scal times a vector
	void addX(float scal, const pVect &p)
	{x+=scal*p.x; y+=scal*p.y; z+=scal*p.z;};

  // product by component
  void prodComp(const pVect &V1, const pVect &V2)
  { x=V1.x*V2.x; y=V1.y*V2.y; z=V1.z*V2.z; };

  // product by component
  void prodComp(const pVect &V)
  { x*=V.x; y*=V.y; z*=V.z; };

  // Update bounding box min-value given a 3D point
  void updateMinBox(const pVect &P);

    // update bounding box max-value given a 3D point
	void updateMaxBox(const pVect &P);

	// clamps each coord value to min "val"
	void clampMin(const float val)
	{x=(x>=val)?x:val; y=(y>=val)?y:val; z=(z>=val)?z:val;};

	// clamps each coord value to max "val"
	void clampMax(const float val)
	{x=(x<=val)?x:val; y=(y<=val)?y:val; z=(z<=val)?z:val;};

	// reverts the vector => opposite
	void revert()
	{x=-x; y=-y; z=-z;};

	// returns the minimum value of x,y,z
	float minCoord() const
	{return ((x<y)?((x<z)?x:z):(y<z)?y:z);};

	// returns the maximum value of x,y,z
	float maxCoord() const
	{return ((x>y)?((x>z)?x:z):(y>z)?y:z);};

  // returns the sum x+y+z
  float sumCoords() const
  {return x+y+z;};

	// "this" coords should be < v coords => reverts vMin and vMax values
	void sortCoordsMinMax(pVect &vMin,pVect &vMax) const;

	// Returns the triangle area
	static float getTriangleArea(const pVect &P1,const pVect &P2,const pVect &P3);

	// creates a point on a sphere, given theta/phi/r
	void setSphereVertex(float th, float ph, const pVect &C, float r);

  // construct u,v basis vectors from *this (normal)
  void setBasisVectors(pVect &u, pVect &v) const;

	// rotate dir vector according to new coord system
  void rotateVectorSphere(const pVect &n, const pVect &u, const pVect &v);

};


//=================================================================
// Matrix operations
//
class matrix {

private:
  pVect L1, L2, L3;   // 3 row vectors

public:
  matrix() :L1(1,0,0),L2(0,1,0),L3(0,0,1) {};
  matrix(const pVect &l1, const pVect &l2, const pVect &l3);
  void setRotation(const pVect &u, const pVect &v, const pVect &w);
  void setRotationX(float A);
  void setRotationY(float A);
  void setRotationZ(float A);
  void transpose();
  matrix operator*(const matrix &M);
  pVect operator*(const pVect &V);

};



#endif //_point_h_



